﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Net.Sockets;
using System.Net;
using Communication.IO.Tools;
using System.Threading;
using System.Reflection;
using System.Net.NetworkInformation;
using System.IO;

namespace Argos_3D_P100_Visualizer
{
    public partial class Argos3dP100VisualizerForm : Form
    {

        public Argos3dP100VisualizerForm()
        {
            System.Reflection.Assembly a = System.Reflection.Assembly.GetExecutingAssembly();
            if (Properties.Settings.Default.ApplicationVersion != a.GetName().Version.ToString())
            {
                Properties.Settings.Default.Upgrade();
                Properties.Settings.Default.ApplicationVersion = a.GetName().Version.ToString();
            }
            InitializeComponent();
            tbChannel1RangeMin.Text = Properties.Settings.Default.range_min_ch1;
            tbChannel1RangeMax.Text = Properties.Settings.Default.range_max_ch1;
            tbChannel2RangeMin.Text = Properties.Settings.Default.range_min_ch2;
            tbChannel2RangeMax.Text = Properties.Settings.Default.range_max_ch2;
            txtStatsFrameCount.Text = Properties.Settings.Default.buffer_length_stat;
            tbCustomRegAddr.Text = Properties.Settings.Default.reg_addr;
            txtSrcProcCmd.Text = Properties.Settings.Default.src_proc_cmd;
            linkLabel1.Links.Add(0, linkLabel1.Text.Length, "dummy");
            lbSrcProcCmdResult.Text = "";
            pmd_sdk = null;
            txtStatsFrameCount_TextChanged(null, null);
            timerConnectionTogglePmdSdk.Interval = 200;
            timerConnectionTogglePmdSdk.Enabled = true;
            Argos3dP100VisualizerForm_Resize(null, null);
            sourceDataHandlerThread = new Thread(sourceDataHandler);
            sourceDataHandlerThread.Name = "SourceDataHandler";
            sourceDataHandlerThread.Start();
            frameHandlerThread = new Thread(frameHandler);
            frameHandlerThread.Name = "FrameHandler";
            frameHandlerThread.Start();
            this.Text += " V" + Assembly.GetExecutingAssembly().GetName().Version.ToString(3);
            timerFrameRate.Enabled = true;
        }


        // Private members for control interface communication

        private PmdSdk pmd_sdk;
        private Statistics statistics_ch1 = new Statistics(1);
        private Statistics statistics_ch2 = new Statistics(1);

        private const UInt16 REG_INTEGRATION_TIME = 5;
        private const UInt16 REG_FRAME_RATE = 10;

        private const int LED_TEMP_THRESH_1 = 60;
        private const int LED_TEMP_THRESH_2 = 80;


        private int framesReceivedCounter = 0;
        private int framesReceivedCounterTemp = 0;

        private int framesDrawnCounter = 0;
        private int framesDrawnCounterTemp = 0;

        private int framesNotDrawnCounter = 0;
        private int mouse_pos_x = -1, mouse_pos_y = -1;
        private bool temperature_exceeded = false;

        private Bitmap imgDist, imgAmp;

        int offsetDist = 0, scaleDist = 3500;
        int offsetAmp = 0, scaleAmp = 2500;

        private Thread streamReceiverThread;
        private Thread sourceDataHandlerThread;
        private Thread frameHandlerThread;
        private Semaphore frameQueueSem = new Semaphore(0, int.MaxValue);
        private Queue<FrameReceivedArgs> frameQueue = new Queue<FrameReceivedArgs>();
        private Semaphore sourceDataQueueSem = new Semaphore(0, int.MaxValue);
        private Queue<PmdSdk.SourceDataSTRUCT> sourceDataQueue = new Queue<PmdSdk.SourceDataSTRUCT>();
        private bool abortStreamReceiverThread = false;


        // This struct holds the data from the data interface header
        // The frame queue consists of intances of this struct
        // All information neccesary to draw and print the screen is included
        public struct FrameReceivedArgs
        {
            public List<List<double>> data;
            public UInt16 xRes;
            public UInt16 yRes;
            public UInt16 dataFormat;
            public UInt32 timeStamp;
            public UInt16 frameCounter;
            public double mainTemp;
            public double ledTemp;
        }


        // Function runnung in a separate thread
        // It repeatedly reads data and enqueues the resulting data frame into the frame queue
        private void streamReceiver()
        {
            abortStreamReceiverThread = false;
            while (!abortStreamReceiverThread)
            {
                try
                {
                    PmdSdk.SourceDataSTRUCT source_data_struct = new PmdSdk.SourceDataSTRUCT();
                    source_data_struct.descr = pmd_sdk.Update();
                    source_data_struct.sourceData = pmd_sdk.getSourceData();
                    framesReceivedCounter++;
                    if (sourceDataQueue.Count <= 5)
                    {
                        sourceDataQueue.Enqueue(source_data_struct);
                        sourceDataQueueSem.Release();
                    }
                }
                catch (Exception exc)
                {
                    Invoke(new MethodInvoker(delegate()
                        {
                            lbConnectionState.Text = exc.Message;
                        }));
                    Thread.Sleep(100);
                }
            }
        }


        private void sourceDataHandler()
        {
            while (true)
            {
                sourceDataQueueSem.WaitOne();
                while (sourceDataQueue.Count > 0)
                {
                    PmdSdk.SourceDataSTRUCT source_data = sourceDataQueue.Dequeue();
                    List<PmdSdk.ContainerSTRUCT> containers = pmd_sdk.getData(source_data, true);
                    if (frameQueue.Count <= 5)
                    {
                        FrameReceivedArgs frame = new FrameReceivedArgs();
                        if (containers[0].header != null)
                        {
                            if (containers[0].header.ContainsKey(PmdSdk.ContainerHeaderENUM.TimeStamp))
                            {
                                frame.timeStamp = containers[0].header[PmdSdk.ContainerHeaderENUM.TimeStamp] * 1000;
                            }
                            if (containers[0].header.ContainsKey(PmdSdk.ContainerHeaderENUM.FrameCounter))
                            {
                                frame.frameCounter = (ushort)containers[0].header[PmdSdk.ContainerHeaderENUM.FrameCounter];
                            }
                            if (containers[0].header.ContainsKey(PmdSdk.ContainerHeaderENUM.TemperatureIllumination))
                            {
                                frame.ledTemp = Math.Round(containers[0].header[PmdSdk.ContainerHeaderENUM.TemperatureIllumination] / 16.0, 2);
                            }
                        }
                        frame.data = new List<List<double>>();
                        frame.xRes = (ushort)source_data.descr.img.numColumns;
                        frame.yRes = (ushort)source_data.descr.img.numRows;
                        frame.dataFormat = (ushort)PmdSdk.ImageDataFormatModes.DistAmp;
                        foreach (PmdSdk.ContainerSTRUCT container in containers)
                        {
                            if (container.data != null)
                            {
                                frame.data.Add(container.data);
                            }
                        }
                        if (frame.data.Count == 2)
                        {
                            frameQueue.Enqueue(frame);
                            frameQueueSem.Release();
                        }
                        // Statistics
                        int pixel_index = mouse_pos_x + mouse_pos_y * frame.xRes;
                        if (pixel_index >= 0 && pixel_index < frame.xRes * frame.yRes)
                        {
                            if ((frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.DistAmp || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.XYZcoordinates || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.XYZamp || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.DistAmpSourceData)
                            {
                                // channel 1 is Dist
                                statistics_ch1.AddElement(frame.data[0], frame.xRes, frame.yRes);
                                if (statistics_ch1.FillSize == statistics_ch1.MaxSize)
                                {
                                    List<double> avgs = statistics_ch1.GetAverage();
                                    if (avgs.Count > pixel_index)
                                    {
                                        Invoke(new MethodInvoker(delegate()
                                        {
                                            lbAvgCh1.Text = avgs[pixel_index].ToString("F1");
                                        }));
                                        List<double> devs = statistics_ch1.GetStdDeviation(avgs);
                                        if (devs.Count > pixel_index)
                                        {
                                            Invoke(new MethodInvoker(delegate()
                                            {
                                                lbStddevCh1.Text = devs[pixel_index].ToString("F1");
                                            }));
                                        }
                                        statistics_ch1.Reset();
                                    }
                                }
                            }
                            if ((frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.DistAmp || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.XYZamp || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.DistAmpSourceData)
                            {
                                // channel 2 is Amp
                                int channelIdx = 1;
                                statistics_ch2.AddElement(frame.data[channelIdx], frame.xRes, frame.yRes);
                                if (statistics_ch2.FillSize == statistics_ch2.MaxSize)
                                {
                                    List<double> avgs = statistics_ch2.GetAverage();
                                    if (avgs.Count > pixel_index)
                                    {
                                        Invoke(new MethodInvoker(delegate()
                                        {
                                            lbAvgCh2.Text = avgs[pixel_index].ToString("F1");
                                        }));
                                        List<double> devs = statistics_ch2.GetStdDeviation(avgs);
                                        if (devs.Count > pixel_index)
                                        {
                                            Invoke(new MethodInvoker(delegate()
                                            {
                                                lbStddevCh2.Text = devs[pixel_index].ToString("F1");
                                            }));
                                        }
                                        else
                                        {
                                        }
                                        statistics_ch2.Reset();
                                    }
                                    else
                                    {
                                    }
                                }
                            }
                        }
                    }
                    else
                    {
                        framesNotDrawnCounter++;
                    }
                }
            }
        }


        // This function runs in a seperate thread
        // It waits for data from the frame queue and transforms it into two bitmaps which are then displayed in the
        // form window along with some information in text form
        private void frameHandler()
        {
            while (true)
            {
                frameQueueSem.WaitOne();
                while (frameQueue.Count > 0)
                {
                    string msg_curr_val_ch1 = "n.a. 0/" + statistics_ch1.MaxSize;
                    string msg_curr_val_ch2 = "n.a. 0/" + statistics_ch2.MaxSize;
                    FrameReceivedArgs frame = frameQueue.Dequeue();
                    if (frame.xRes <= 0 || frame.yRes <= 0)
                    {
                        continue;
                    }
                    if (imgDist == null)
                    {
                        imgDist = new Bitmap(frame.xRes, frame.yRes, System.Drawing.Imaging.PixelFormat.Format24bppRgb);
                    }
                    if (imgAmp == null)
                    {
                        imgAmp = new Bitmap(frame.xRes, frame.yRes, System.Drawing.Imaging.PixelFormat.Format24bppRgb);
                    }
                    List<byte> img_dist_rgb = new List<byte>();
                    List<byte> img_amp_rgb = new List<byte>();
                    if ((frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.DistAmp || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.XYZcoordinates || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.XYZamp || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.DistAmpSourceData)
                    {
                        // channel 1 is Dist
                        if (mouse_pos_x > 0 && mouse_pos_y > 0 && mouse_pos_x < frame.xRes && mouse_pos_y < frame.yRes)
                        {
                            msg_curr_val_ch1 = frame.data[0][mouse_pos_x + mouse_pos_y * frame.xRes].ToString("F1");
                        }
                        int index = 0;
                        for (int y = 0; y < frame.yRes; y++)
                        {
                            for (int x = 0; x < frame.xRes; x++)
                            {
                                byte r, g, b;
                                if (frame.data[0][index] == 0)
                                {
                                    r = 0;// byte.MaxValue;
                                    g = 0;// byte.MaxValue;
                                    b = 0;// byte.MaxValue;
                                }
                                else if (frame.data[0][index] == 0xffff)
                                {
                                    r = 0;
                                    g = 0;
                                    b = 0;
                                }
                                else
                                {
                                    long pix = Convert.ToInt64((frame.data[0][index] - offsetDist) * UInt16.MaxValue / scaleDist);
                                    if (pix < UInt16.MinValue)
                                    {
                                        pix = UInt16.MinValue;
                                    }
                                    else if (pix > UInt16.MaxValue)
                                    {
                                        pix = UInt16.MaxValue;
                                    }

                                    if (pix <= UInt16.MaxValue / 4)
                                    {
                                        r = byte.MaxValue;
                                        g = (byte)(pix * byte.MaxValue / (UInt16.MaxValue / 4));
                                        b = 0;
                                    }
                                    else
                                    {
                                        pix -= UInt16.MaxValue / 4;
                                        if (pix <= UInt16.MaxValue / 4)
                                        {
                                            r = (byte)(byte.MaxValue - pix * byte.MaxValue / (UInt16.MaxValue / 4));
                                            g = byte.MaxValue;
                                            b = 0;
                                        }
                                        else
                                        {
                                            pix -= UInt16.MaxValue / 4;
                                            if (pix <= UInt16.MaxValue / 4)
                                            {
                                                r = 0;
                                                g = byte.MaxValue;
                                                b = (byte)(pix * byte.MaxValue / (UInt16.MaxValue / 4));
                                            }
                                            else
                                            {
                                                pix -= UInt16.MaxValue / 4;
                                                r = 0;
                                                g = (byte)(byte.MaxValue - pix * byte.MaxValue / (UInt16.MaxValue / 4));
                                                b = byte.MaxValue;
                                            }
                                        }
                                    }
                                }
                                img_dist_rgb.Add(b);
                                img_dist_rgb.Add(g);
                                img_dist_rgb.Add(r);
                                index++;
                            }
                        }
                    }
                    if ((frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.DistAmp || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.XYZamp || (frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.DistAmpSourceData)
                    {
                        // channel 2 is Amp
                        int channelIdx = 1;
                        if ((frame.dataFormat & 0xfff8) == (int)PmdSdk.ImageDataFormatModes.XYZamp)
                        {
                            channelIdx = 1;
                        }
                        if (mouse_pos_x > 0 && mouse_pos_y > 0 && mouse_pos_x < frame.xRes && mouse_pos_y < frame.yRes)
                        {
                            msg_curr_val_ch2 = frame.data[channelIdx][mouse_pos_x + mouse_pos_y * frame.xRes].ToString();
                        }
                        int index = 0;
                        for (int y = 0; y < frame.yRes; y++)
                        {
                            for (int x = 0; x < frame.xRes; x++)
                            {
                                long ampScaled = Convert.ToInt64((frame.data[channelIdx][index] - offsetAmp) * UInt16.MaxValue / scaleAmp);
                                byte brightness;
                                if (ampScaled >= UInt16.MaxValue)
                                {
                                    brightness = byte.MaxValue;
                                }
                                else if (ampScaled <= UInt16.MinValue)
                                {
                                    brightness = byte.MinValue;
                                }
                                else
                                {
                                    brightness = (byte)(ampScaled >> 8);
                                }
                                img_amp_rgb.Add(brightness);
                                img_amp_rgb.Add(brightness);
                                img_amp_rgb.Add(brightness);
                                index++;
                            }
                        }
                    }

                    try
                    {
                        // copy first bitmap data
                        System.Drawing.Imaging.BitmapData bmpData = imgDist.LockBits(new Rectangle(0, 0, imgDist.Width, imgDist.Height), System.Drawing.Imaging.ImageLockMode.ReadWrite, imgDist.PixelFormat);
                        IntPtr ptr = bmpData.Scan0;
                        System.Runtime.InteropServices.Marshal.Copy(img_dist_rgb.ToArray(), 0, bmpData.Scan0, img_dist_rgb.Count);
                        imgDist.UnlockBits(bmpData);

                        // copy second bitmap data
                        bmpData = imgAmp.LockBits(new Rectangle(0, 0, imgAmp.Width, imgAmp.Height), System.Drawing.Imaging.ImageLockMode.ReadWrite, imgAmp.PixelFormat);
                        ptr = bmpData.Scan0;
                        System.Runtime.InteropServices.Marshal.Copy(img_amp_rgb.ToArray(), 0, bmpData.Scan0, img_amp_rgb.Count);
                        imgAmp.UnlockBits(bmpData);

                        framesDrawnCounter++;
                        Invoke(new publishDelegate(publish), new object[] { imgDist, imgAmp, frame.dataFormat, frame.timeStamp, frame.frameCounter, Convert.ToDouble(frame.mainTemp), Convert.ToDouble(frame.ledTemp), msg_curr_val_ch1, msg_curr_val_ch2 });
                    }
                    catch (Exception exc)
                    {
                        imgDist = null;
                        imgAmp = null;
                    }
                }
            }
        }



        // This function is invoked, in order to be able to access the visual elements
        // The information in the function's arguments are the displayed in the main form
        private delegate void publishDelegate(Bitmap leftImage, Bitmap rightImage, UInt16 dataFormat, UInt32 timestamp, UInt16 frameCounter, double mainTemp, double ledTemp, string msg_curr_val_ch1, string msg_curr_val_ch2);
        private void publish(Bitmap leftImage, Bitmap rightImage, UInt16 dataFormat, UInt32 timestamp, UInt16 frameCounter, double mainTemp, double ledTemp, string msg_curr_val_ch1, string msg_curr_val_ch2)
        {
            pbChannel1Image.Image = new Bitmap(leftImage);
            pbChannel2Image.Image = new Bitmap(rightImage);
            if (dataFormat >> 3 == 0)
            {
                lbChannel1Name.Text = "Dist [mm]";
                lbChannel2Name.Text = "Amp";
                lbCurrValCh1Label.Text = "Dist [mm]:";
                lbCurrValCh2Label.Text = "Amp:";
            }
            else
            {
                lbChannel1Name.Text = "N.a.";
                lbChannel2Name.Text = "N.a.";
            }
            lblFrameCounter.Text = frameCounter.ToString();
            lblTemperature.Text = ledTemp.ToString("F1") + " / " + (ledTemp*1.8 + 32).ToString("F1");
            if (ledTemp >= LED_TEMP_THRESH_2)
            {
                lblTemperature.ForeColor = Color.Red;
                lblTemperature.Font = new Font(lblTemperature.Font, FontStyle.Bold);
                label10.ForeColor = Color.Red;
                //label10.Font = new Font(label10.Font, FontStyle.Bold);
                if (!temperature_exceeded)
                {
                    temperature_exceeded = true;
                    string msg = "The temperature is dangerously high." + Environment.NewLine;
                    msg += "Automatically setting the integration time to a save value.";
                    if (MessageBox.Show(msg, "Warning", MessageBoxButtons.OKCancel, MessageBoxIcon.Warning) == DialogResult.OK)
                    {
                        try
                        {
                            pmd_sdk.setRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0IntTime, 100);
                        }
                        catch (Exception exc)
                        {
                            MessageBox.Show("Failed to set integration time." + Environment.NewLine + exc.Message);
                        }
                    }
                }
            }
            else if (ledTemp >= LED_TEMP_THRESH_1)
            {
                if (ledTemp < LED_TEMP_THRESH_2 - 1)  //'hysteresis'
                {
                    temperature_exceeded = false;
                }
                lblTemperature.ForeColor = Color.Orange;
                lblTemperature.Font = new Font(lblTemperature.Font, FontStyle.Bold);
                label10.ForeColor = Color.Orange;
                //label10.Font = new Font(label10.Font, FontStyle.Bold);
            }
            else
            {
                lblTemperature.ForeColor = SystemColors.ControlText;
                lblTemperature.Font = new Font(lblTemperature.Font, 0);
                label10.ForeColor = SystemColors.ControlText;
                //label10.Font = new Font(label10.Font, 0);
            }
            if (mouse_pos_x > 0 && mouse_pos_y > 0 && mouse_pos_x < leftImage.Width && mouse_pos_y < leftImage.Height)
            {
                lbStatisticsPos.Text = "(x, y): (" + mouse_pos_x + ", " + mouse_pos_y + ")";
            }
            else
            {
                lbStatisticsPos.Text = "(x, y): n.a.";
            }
            lbCurrValCh1.Text = msg_curr_val_ch1;
            lbCurrValCh2.Text = msg_curr_val_ch2;
            lblStatisticsQueueCount.Text = statistics_ch1.FillSize + "/" + statistics_ch1.MaxSize;
        }


        #region ClickEventHandler
        private void bttnTintSet_Click(object sender, EventArgs e)
        {
            long valueToWrite;
            try
            {
                valueToWrite = Convert.ToUInt16(tbTint.Text);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid integration time, please");
                return;
            }
            try
            {
                int phase_time = (int) pmd_sdk.getRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0FrameTime);
                if ((phase_time != 0) && (phase_time < valueToWrite + 1500))
                {
                    MessageBox.Show("Maximum integration time for a phase time of " + phase_time + "µs is " + (phase_time - 1500).ToString() + "µs", "Conflict", MessageBoxButtons.OK, MessageBoxIcon.Stop);
                    return;
                }
                pmd_sdk.setRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0IntTime, valueToWrite);
                tbTint.BackColor = Color.Green;
            }
            catch (Exception exc)
            {
                tbTint.BackColor = Color.Red;
            }
            timerTintColor.Enabled = false;
            timerTintColor.Interval = 1000;
            timerTintColor.Enabled = true;
        }


        private void bttnTintGet_Click(object sender, EventArgs e)
        {
            long regValue;
            try
            {
                regValue = pmd_sdk.getRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0IntTime);
                tbTint.BackColor = Color.Green;
                tbTint.Text = Convert.ToString(regValue);
            }
            catch (Exception exc)
            {
                tbTint.BackColor = Color.Red;
            }
            timerTintColor.Enabled = false;
            timerTintColor.Interval = 1000;
            timerTintColor.Enabled = true;
        }


        private void bttnFrameRateSet_Click(object sender, EventArgs e)
        {
            long valueToWrite;
            try
            {
                valueToWrite = Convert.ToInt64(Math.Ceiling((1000000.0 / Convert.ToUInt16(tbFrameRate.Text)) / 4.0));
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid frame rate, please");
                return;
            }
            try
            {
                int t_int = (int)pmd_sdk.getRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0IntTime);
                if (valueToWrite < t_int + 1500)
                {
                    MessageBox.Show("Maximum frame rate for an integration time of " + t_int + "µs is " + (1000000 / (4 * (t_int + 1500))).ToString() + "Hz", "Conflict", MessageBoxButtons.OK, MessageBoxIcon.Stop);
                    return;
                }
                pmd_sdk.setRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0FrameTime, valueToWrite);
                tbFrameRate.BackColor = Color.Green;
            }
            catch (Exception exc)
            {
                tbFrameRate.BackColor = Color.Red;
            }
            timerFrameRateColor.Enabled = false;
            timerFrameRateColor.Interval = 1000;
            timerFrameRateColor.Enabled = true;
        }


        private void bttnPhaseTimeSet_Click(object sender, EventArgs e)
        {
            long valueToWrite;
            try
            {
                valueToWrite = Convert.ToUInt16(tbPhaseTime.Text);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid frame time, please");
                return;
            }
            try
            {
                int t_int = (int)pmd_sdk.getRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0IntTime);
                if ((valueToWrite < t_int + 1500) && (valueToWrite != 0))
                {
                    MessageBox.Show("Minimum Phase time for an integration time of " + t_int + "µs is " + (t_int + 1500).ToString() + "µs (or 0µs)", "Conflict", MessageBoxButtons.OK, MessageBoxIcon.Stop);
                    return;
                }
                pmd_sdk.setRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0FrameTime, valueToWrite);
                tbPhaseTime.BackColor = Color.Green;
            }
            catch (Exception exc)
            {
                tbPhaseTime.BackColor = Color.Red;
            }
            timerFrameRateColor.Enabled = false;
            timerFrameRateColor.Interval = 1000;
            timerFrameRateColor.Enabled = true;
        }


        private void bttnFrameRateGet_Click(object sender, EventArgs e)
        {
            long regValue;
            try
            {
                regValue = pmd_sdk.getRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0FrameTime);
                tbFrameRate.BackColor = Color.Green;
                if (regValue != 0)
                {
                    tbFrameRate.Text = Convert.ToString(1000000 / (4 * regValue));
                }
                else
                {
                    int t_int = (int)pmd_sdk.getRegister(PmdSdk.AddressRegistersPmdSdk.Sequence0IntTime);
                    tbFrameRate.Text = (250000/(t_int + 1500)).ToString();
                }
                tbPhaseTime.BackColor = Color.Green;
                tbPhaseTime.Text = Convert.ToString(regValue);
            }
            catch (Exception exc)
            {
                tbFrameRate.BackColor = Color.Red;
                tbPhaseTime.BackColor = Color.Red;
            }
            timerFrameRateColor.Enabled = false;
            timerFrameRateColor.Interval = 1000;
            timerFrameRateColor.Enabled = true;
        }


        private void bttnCustomRegSet_Click(object sender, EventArgs e)
        {
            long addrToWrite;
            try
            {
                addrToWrite = Convert.ToUInt16(tbCustomRegAddr.Text, (int)16);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid register address, please");
                return;
            }

            long valueToWrite;
            try
            {
                valueToWrite = Convert.ToUInt16(tbCustomRegValue.Text);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid register value, please");
                return;
            }

            try
            {
                pmd_sdk.setRegister((PmdSdk.AddressRegistersPmdSdk)addrToWrite, valueToWrite);
                tbCustomRegValue.BackColor = Color.Green;
            }
            catch (Exception exc)
            {
                tbCustomRegValue.BackColor = Color.Red;
            }
            timerCustomRegColor.Enabled = false;
            timerCustomRegColor.Interval = 1000;
            timerCustomRegColor.Enabled = true;
        }


        private void bttnCustomRegGet_Click(object sender, EventArgs e)
        {
            long addrToRead;
            try
            {
                addrToRead = Convert.ToUInt16(tbCustomRegAddr.Text, (int)16);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid register address, please");
                return;
            }

            long regValue;
            try
            {
                regValue = pmd_sdk.getRegister((PmdSdk.AddressRegistersPmdSdk)addrToRead);
                tbCustomRegValue.BackColor = Color.Green;
                tbCustomRegValue.Text = Convert.ToString(regValue);
            }
            catch (Exception exc)
            {
                tbCustomRegValue.BackColor = Color.Red;
            }
            timerCustomRegColor.Enabled = false;
            timerCustomRegColor.Interval = 1000;
            timerCustomRegColor.Enabled = true;
        }


        private void bttnConnectionToggle_Click(object sender, EventArgs e)
        {
            timerConnectionTogglePmdSdk.Interval = 200;
            timerConnectionTogglePmdSdk.Enabled = true;
        }

        delegate void connectionTogglePmdSdkDelegate();
        private void connectionTogglePmdSdk()
        {
            if (pmd_sdk == null)
            {
                try
                {
                    lbConnectionState.Text = "Connecting...";
                    groupDataInterface.Refresh();
                    pmd_sdk = new PmdSdk("digicam", "", "digicamproc", "");
                    streamReceiverThread = new Thread(streamReceiver);
                    streamReceiverThread.Name = "StreamReceiver";
                    streamReceiverThread.Start();
                    bttnConnectionToggle.Text = "Disconnect";
                    lbConnectionState.Text = "Connected: " + pmd_sdk.getDevicePonSn();
                }
                catch (Exception exc)
                {
                    bttnConnectionToggle.Text = "Connect";
                    lbConnectionState.Text = "Could not connect: " + exc.Message;
                }
            }
            else
            {
                lbConnectionState.Text = "Disconnecting...";
                groupDataInterface.Refresh();
                abortStreamReceiverThread = true;
                while (streamReceiverThread.IsAlive) ;
                streamReceiverThread = null;
                try
                {
                    pmd_sdk.Close();
                    lbConnectionState.Text = "Disconnected";
                }
                catch (Exception exc)
                {
                    lbConnectionState.Text = "Connection disrupted: " + exc.Message;
                }
                pmd_sdk = null;
                bttnConnectionToggle.Text = "Connect";
            }
        }

        private void cbImageZoom_CheckedChanged(object sender, EventArgs e)
        {
            if (cbImageZoom.Checked)
            {
                pbChannel1Image.SizeMode = PictureBoxSizeMode.Normal;
                pbChannel2Image.SizeMode = PictureBoxSizeMode.Normal;
            }
            else
            {
                pbChannel1Image.SizeMode = PictureBoxSizeMode.Zoom;
                pbChannel2Image.SizeMode = PictureBoxSizeMode.Zoom;
            }
        }

        private void bttnSrcCmd_Click(object sender, EventArgs e)
        {
            if (txtSrcProcCmd.Text == "this is awesome")
            {
                MessageBox.Show("Thank you! You're very smart.");
                return;
            }
            try
            {
                pmd_sdk.execSourceCmd(txtSrcProcCmd.Text);
            }
            catch (Exception exc)
            {
                MessageBox.Show(exc.Message);
            }
            lbSrcProcCmdResult.Text = pmd_sdk.getStatusMsg();
        }

        private void bttnProcCmd_Click(object sender, EventArgs e)
        {
            if (txtSrcProcCmd.Text == "this is awesome")
            {
                MessageBox.Show("Thank you! You're very smart.");
                return;
            }
            try
            {
                pmd_sdk.execProcessingCmd(txtSrcProcCmd.Text);
            }
            catch (Exception exc)
            {
                MessageBox.Show(exc.Message);
            }
            lbSrcProcCmdResult.Text = pmd_sdk.getStatusMsg();
        }
        #endregion


        #region GeneralEventHandler
        private void Argos3dP100VisualizerForm_Resize(object sender, EventArgs e)
        {
            pbChannel1Image.Width = this.Size.Width / 2 - 15;
            pbChannel1Image.Height = pbChannel1Image.Width * 3 / 4;
            pbChannel1Image.Left = 5;
            pbChannel2Image.Width = this.Size.Width / 2 - 15;
            pbChannel2Image.Height = pbChannel2Image.Width * 3 / 4;
            pbChannel2Image.Left = this.Size.Width / 2 - 5;
        }


        private void Argos3dP100VisualizerForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            sourceDataHandlerThread.Abort();
            if (pmd_sdk != null)
            {
                connectionTogglePmdSdk();
            }
            frameHandlerThread.Abort();
            if (streamReceiverThread != null)
            {
                abortStreamReceiverThread = true;
                while (streamReceiverThread.IsAlive) ;
            }
        }


        private void timerTintColor_Tick(object sender, EventArgs e)
        {
            tbTint.BackColor = Color.White;
            timerTintColor.Enabled = false;
        }


        private void timerFrameRate_Tick(object sender, EventArgs e)
        {
            if (Properties.Settings.Default.first_start)
            {
                Properties.Settings.Default.first_start = false;
                Properties.Settings.Default.Save();
                pictureBox1_Click(null, null);
            }
            int frameRateDrawn = framesDrawnCounter - framesDrawnCounterTemp;
            framesDrawnCounterTemp = framesDrawnCounter;
            int frameRateReceived = framesReceivedCounter - framesReceivedCounterTemp;
            framesReceivedCounterTemp = framesReceivedCounter;
            Invoke(new MethodInvoker(delegate()
            {
                lblFramerateDrawn.Text = frameRateDrawn.ToString();
                lblFramerateRecv.Text = frameRateReceived.ToString();
            }));
        }


        private void timerFrameRateColor_Tick(object sender, EventArgs e)
        {
            tbFrameRate.BackColor = Color.White;
            tbPhaseTime.BackColor = Color.White;
            timerFrameRateColor.Enabled = false;
        }

        private void timerCustomRegColor_Tick(object sender, EventArgs e)
        {
            tbCustomRegValue.BackColor = Color.White;
            timerCustomRegColor.Enabled = false;
        }

        private void timerConnectionTogglePmdSdk_Tick(object sender, EventArgs e)
        {
            connectionTogglePmdSdk();
            timerConnectionTogglePmdSdk.Enabled = false;
        }

        private void pbChannel1Image_MouseLeave(object sender, EventArgs e)
        {
            mouse_pos_x = -1;
            mouse_pos_y = -1;
        }

        private void pbChannel1Image_MouseMove(object sender, MouseEventArgs e)
        {
            int mouse_pos_x_before = mouse_pos_x;
            int mouse_pos_y_before = mouse_pos_y;
            try
            {
                PictureBox pb = (PictureBox)sender;
                if (pb.Image == null)
                {
                    return;
                }
                if (cbImageZoom.Checked)
                {
                    //Point point = pb.PointToClient(new Point(e.X, e.Y));
                    mouse_pos_x = e.X;
                    mouse_pos_y = e.Y;
                    if (mouse_pos_x < 0 || mouse_pos_x >= pb.Image.Width || mouse_pos_y < 0 || mouse_pos_y >= pb.Image.Height)
                    {
                        mouse_pos_x = -1;
                        mouse_pos_y = -1;
                    }
                    return;
                }
                double scale = (double)pb.Image.Width / (double)pb.Image.Height;
                int width = 0;
                int height = 0;
                if (((double)pb.Height * scale) > (double)pb.Width)
                {
                    width = pb.Width;
                    height = (int)((double)pb.Width / scale);
                }
                else
                {
                    height = pb.Height;
                    width = (int)((double)pb.Height * scale);
                }

                double x = (double)e.X - (((double)(pb.Width - width) / 2.0));
                double y = (double)e.Y - (((double)(pb.Height - height) / 2.0));

                scale = (double)pb.Image.Width / (double)width;
                mouse_pos_x = (int)(x * scale) + 1;
                mouse_pos_y = (int)(y * scale) + 1;
            }
            catch (Exception exc)
            {
                mouse_pos_x = -1;
                mouse_pos_y = -1;
            }
            if (mouse_pos_x != mouse_pos_x_before || mouse_pos_y != mouse_pos_y_before)
            {
                statistics_ch1.Reset();
                statistics_ch2.Reset();
                Invoke(new MethodInvoker(delegate()
                {
                    lbAvgCh1.Text = "n.a.";
                    lbStddevCh1.Text = "n.a.";
                    lbAvgCh2.Text = "n.a.";
                    lbStddevCh2.Text = "n.a.";
                }));
            }
        }

        private void txtStatsFrameCount_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.buffer_length_stat = txtStatsFrameCount.Text;
            Properties.Settings.Default.Save();
            int frame_count = 1;
            try
            {
                frame_count = Convert.ToInt32(txtStatsFrameCount.Text);
                txtStatsFrameCount.BackColor = Color.White;
            }
            catch
            {
                txtStatsFrameCount.BackColor = Color.Orange;
            }
            if (frame_count < 1)
            {
                frame_count = 1;
                txtStatsFrameCount.BackColor = Color.Orange;
            }
            statistics_ch1.MaxSize = frame_count;
            statistics_ch2.MaxSize = frame_count;
        }
        #endregion

        private void linkLabel1_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            System.Diagnostics.Process.Start("https://support.bluetechnix.at/wiki/PMDSDK_/_PMDMDK_User_Manual");
        }

        private void pictureBox1_Click(object sender, EventArgs e)
        {
            string msg = "";
            msg += "Argos3D - P100 Visualizer V" + Assembly.GetExecutingAssembly().GetName().Version.ToString(3) + Environment.NewLine;
            msg += "Build: " + Assembly.GetExecutingAssembly().GetName().Version.ToString() + Environment.NewLine;
            msg += Environment.NewLine;
            msg += "© Copyright Bluetechnix GmbH 2013" + Environment.NewLine;
            msg += "" + Environment.NewLine;
            msg += "No warranty" + Environment.NewLine;
            msg += "This program is distributed as it is, without warranty of any kind, either expressed or implied, including, ";
            msg += "but not limited to, the implied warranties of merchantability and fitness for a particular purpose." + Environment.NewLine;
            msg += Environment.NewLine;
            msg += "www.bluetechnix.com" + Environment.NewLine;
            msg += Environment.NewLine;
            msg += "Click on the Bluetechnix logo in order to see this information again.";
            MessageBox.Show(msg, "About this software", MessageBoxButtons.OK, MessageBoxIcon.Information, MessageBoxDefaultButton.Button1, 0, "http://bluetechnix.com");
        }

        private void txtSrcProcCmd_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.src_proc_cmd = txtSrcProcCmd.Text;
            Properties.Settings.Default.Save();
        }

        private void tbChannel1RangeMin_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.range_min_ch1 = tbChannel1RangeMin.Text;
            Properties.Settings.Default.Save();
            try
            {
                offsetDist = Convert.ToInt32(tbChannel1RangeMin.Text);
                tbChannel1RangeMin.BackColor = Color.White;
            }
            catch
            {
                offsetDist = 0;
                tbChannel1RangeMin.BackColor = Color.Orange;
            }
            if (offsetDist < 0)
            {
                offsetDist = 0;
                tbChannel1RangeMin.BackColor = Color.Orange;
            }
            tbChannel1RangeMax_TextChanged(null, null);
        }

        private void tbChannel1RangeMax_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.range_max_ch1 = tbChannel1RangeMax.Text;
            Properties.Settings.Default.Save();
            try
            {
                scaleDist = Convert.ToInt32(tbChannel1RangeMax.Text) - offsetDist;
                tbChannel1RangeMax.BackColor = Color.White;
            }
            catch
            {
                scaleDist = 3500;
                tbChannel1RangeMax.BackColor = Color.Orange;
            }
            if (scaleDist <= 0)
            {
                scaleDist = 3500;
                tbChannel1RangeMax.BackColor = Color.Orange;
            }
        }

        private void tbChannel2RangeMin_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.range_min_ch2 = tbChannel2RangeMin.Text;
            Properties.Settings.Default.Save();
            try
            {
                offsetAmp = Convert.ToInt32(tbChannel2RangeMin.Text);
                tbChannel2RangeMin.BackColor = Color.White;
            }
            catch
            {
                offsetAmp = 0;
                tbChannel2RangeMin.BackColor = Color.Orange;
            }
            if (offsetAmp < 0)
            {
                offsetAmp = 0;
                tbChannel2RangeMin.BackColor = Color.Orange;
            }
            tbChannel2RangeMax_TextChanged(null, null);
        }

        private void tbChannel2RangeMax_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.range_max_ch2 = tbChannel2RangeMax.Text;
            Properties.Settings.Default.Save();
            try
            {
                scaleAmp = Convert.ToInt32(tbChannel2RangeMax.Text) - offsetAmp;
                tbChannel2RangeMax.BackColor = Color.White;
            }
            catch
            {
                scaleAmp = 3500;
                tbChannel2RangeMax.BackColor = Color.Orange;
            }
            if (scaleAmp <= 0)
            {
                scaleAmp = 3500;
                tbChannel2RangeMax.BackColor = Color.Orange;
            }
        }

        private void tbCustomRegAddr_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.reg_addr = tbCustomRegAddr.Text;
            Properties.Settings.Default.Save();
        }

    }
}
